 /*
 * Copyright (c) 2022 Green Arrowhead LLP

 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

 
/**
 * This provides methods to read configurable parameters required to 
 * connect to internetmessagebus.com
 * @version 2022-11-11
 */
package com.gah.imb.client;


import java.io.File;
import java.io.FileInputStream;
import java.net.InetAddress;
import java.util.Enumeration;
import java.util.Properties;

/**
 * Gets configuration parameters from the properties file mentioned in the 
 * system property: config_imb. All configuration parameters used by IMBClientMain
 * are mentioned in the configuration file.
 * 
 * @author Green Arrowhead LLP
 * @version 2022-11-01
 */
public class CoreProperties {
	//
	//change by mentioning these in the property file, use values provided by IMB ----
	//------------------
	//
	public String server = ""; //internet message bus server url
	public String port = "";   //internet message bus port
	public String user = "";   //internet message bus user id
	public String pass = "";   //internet message bus password
	//------------------
	
	
	//
	//following 2 variables are part of encryption/decryption,
	//values mentioned in property file should exactly match the values
	//mentioned in index.php file
	//------------------
	//
	public String initialVectorString = ""; // This has to be 16 characters e.g. 0123456789123456
	public String secretKey = ""; //Replace this by your secret key; e.g. ReplaceThisByYouSecretKey
	//------------------
	
	
	//
	//change this by mentioning the values in the property file----
	//------------------
	//Application server url. IMB client will be doing Request/Response
	//with this server.
	//Typically, this will be the url of your application/portal
	public String mainserver = "http://localhost:9999";
	//-------------------

	//
	//no need to change, request/response queue name of IMB server
	//-------------------
	public String requestqueue = "isb.http.request.";
	//-------------------
	
	private Properties properties = new Properties();
	
	//host name on which IMB client is running. This will be set automatically
	//at the time of initialization of IMB client. NO NEED TO CHANGE THIS.
	String hostmachinename = "localhost";
	
	private void readPropertyFile(){
		try {
			Properties p = System.getProperties();
			String fileName = (String)p.get("config_imb");

			if(fileName == null){
				fileName = "../../imb.properties";
				
			}
			Logger.log("fileName: "     + fileName);
			
			//check if file exists
			File propfile = new File(fileName);
			
			if(propfile.exists()) {
				properties.load(new FileInputStream(fileName));
			}
			else {
				Logger.log("Not able to read properties from file as file does not exist");
			}
			
		    Enumeration<String> enums = (Enumeration<String>) properties.propertyNames();
		    while (enums.hasMoreElements()) {
		      String key = enums.nextElement();
		      String value = properties.getProperty(key);
		      if(key.equals("server")) {server = value;}
		      if(key.equals("port")) {port = value;}
		      if(key.equals("user")) {user = value;}
		      if(key.equals("pass")) {pass = value;}
		      if(key.equals("requestqueue")) {requestqueue = value;}
		      if(key.equals("mainserver")) {mainserver = value;}
		      if(key.equals("initialVectorString")) {initialVectorString = value;}
		      if(key.equals("secretKey")) {secretKey = value;}
		    }
				
		} 
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	//read property file
	private void readPropertyFile(String configFileLocation){
		try {
		    properties.load(new FileInputStream(configFileLocation));
		} 
		catch (Exception e) {
	
		}
	}	
	
	//constructor used to initialize the object
	public CoreProperties(){
		initialize();
		
		readPropertyFile();

		
	}

	public CoreProperties(String configFileLocation){
		initialize();
		readPropertyFile(configFileLocation);
		

	}
	
	public void initialize() {
		try{
			hostmachinename = InetAddress.getLocalHost().getHostName();
		}
		catch(Exception exp) {
			exp.printStackTrace();
		}
	}
	
	public String getStringProperty(String propName){
		
		String propertyvalue = null;

		if(properties.containsKey(propName)) {
			propertyvalue = properties.getProperty(propName);
		}
		return propertyvalue;
	}
	
	
	public Properties getProperty(){
		return properties;
	}
	
	
	public static void setSystemProperty(Properties prop){
	
		System.setProperties(prop);
	}

}

